//
//  TimerViewController.swift
//  Clock
//
//  Created by Dinosaur_Weirdo on 10/15/18.
//  Copyright © 2018 DinoW. All rights reserved.
//

import Cocoa
import AVFoundation

struct Timer {
    enum state {
        case on, off, paused
    }
    
    var hours: Int
    var minutes: Int
    var seconds: Int
    var hoursLeft: Int
    var minutesLeft: Int
    var secondsLeft: Int
    var ringtone: String
    var currentState: Timer.state
    
    
    init(hours: Int, minutes: Int, seconds: Int, ringtone: String) {
        self.hours        = hours
        self.minutes      = minutes
        self.seconds      = seconds
        self.hoursLeft    = hours
        self.minutesLeft  = minutes
        self.secondsLeft  = seconds
        self.ringtone     = ringtone
        self.currentState = .off
    }
}

class TimerViewController: NSViewController, NSTextFieldDelegate, NSUserNotificationCenterDelegate {
    
    let defaults = UserDefaults.standard
    let hoursKey = "hour"
    let minutesKey = "minute"
    let secondsKey = "second"
    let ringtoneKey = "ringtone"
    
    var audioPlayer = AVAudioPlayer()
    let ringtonesPath = "/System/Library/PrivateFrameworks/ToneLibrary.framework/Versions/A/Resources/Ringtones/"
    let notificationCenter = AppDelegate.notificationCenter
    
    var hours: Int = 0
    var minutes: Int = 0
    var seconds: Int = 0
    var timeLeft: Int = 0
    var ringtone: String = ""
    var timer: Timer? = nil
    
    var actualTimer: DispatchSourceTimer!
    let queue = DispatchQueue(label: "com.dinow.Clock.timer", qos: .userInteractive)
    
    var hourLabel   = NSTextField(frame: NSRect(x: 90, y: 133, width: 70, height: 17))
    var minuteLabel = NSTextField(frame: NSRect(x: 190, y: 133, width: 70, height: 17))
    var secondLabel = NSTextField(frame: NSRect(x: 290, y: 133, width: 70, height: 17))
    var hourTextField   = NSTextField(frame: NSRect(x:  90, y: 150, width: 70, height: 65))
    var minuteTextField = NSTextField(frame: NSRect(x: 190, y: 150, width: 70, height: 65))
    var secondTextField = NSTextField(frame: NSRect(x: 290, y: 150, width: 70, height: 65))
    var hourStepper   = NSStepper(frame: NSRect(x: 162, y: 150, width: 20, height: 65))
    var minuteStepper = NSStepper(frame: NSRect(x: 262, y: 150, width: 20, height: 65))
    var secondStepper = NSStepper(frame: NSRect(x: 362, y: 150, width: 20, height: 65))
    var ringtonesPopUpButton = NSPopUpButton(frame: NSRect(x: 90, y: 100, width: 70, height: 26))
    
    @IBOutlet weak var startPauseResumeBtn: NSButton!
    @IBOutlet weak var cancelBtn: NSButton!
        
    override func viewDidLoad() {
        super.viewDidLoad()
        // Do view setup here.
        
//        notificationCenter.delegate = self
        
        hours = defaults.integer(forKey: hoursKey)
        minutes = defaults.integer(forKey: minutesKey)
        seconds = defaults.integer(forKey: secondsKey)
        ringtone = defaults.string(forKey: ringtoneKey) ?? "None"
        
        timer = Timer(hours: hours, minutes: minutes, seconds: seconds, ringtone: ringtone)
        cancelBtn.isEnabled = false
        
        hourLabel.stringValue = "hours"
        hourLabel.isBezeled = false
        hourLabel.isEditable = false
        hourLabel.drawsBackground = false
        hourLabel.alignment = .right

        minuteLabel.stringValue = "minutes"
        minuteLabel.isBezeled = false
        minuteLabel.isEditable = false
        minuteLabel.drawsBackground = false
        minuteLabel.alignment = .right

        secondLabel.stringValue = "seconds"
        secondLabel.isBezeled = false
        secondLabel.isEditable = false
        secondLabel.drawsBackground = false
        secondLabel.alignment = .right
        
        hourTextField.delegate = self
        minuteTextField.delegate = self
        secondTextField.delegate = self
        
        hourTextField.font = NSFont.monospacedDigitSystemFont(ofSize: 50.0, weight: .regular)
        minuteTextField.font = NSFont.monospacedDigitSystemFont(ofSize: 50.0, weight: .regular)
        secondTextField.font = NSFont.monospacedDigitSystemFont(ofSize: 50.0, weight: .regular)
        
        hourTextField.alignment = .right
        minuteTextField.alignment = .right
        secondTextField.alignment = .right
        
        hourStepper.action   = #selector(self.hourStepperPressed)
        minuteStepper.action = #selector(self.minuteStepperPressed)
        secondStepper.action = #selector(self.secondStepperPressed)
        
        hourStepper.target = self
        minuteStepper.target = self
        secondStepper.target = self
        
        hourStepper.maxValue = 23
        minuteStepper.maxValue = 59
        secondStepper.maxValue = 59
        
        ringtonesPopUpButton.removeAllItems()
        ringtonesPopUpButton.addItem(withTitle: "None")
        ringtonesPopUpButton.addItems(withTitles: getRingtones(path: ringtonesPath).sorted())
        ringtonesPopUpButton.selectItem(withTitle: ringtone)
        ringtonesPopUpButton.setFrameSize(ringtonesPopUpButton.intrinsicContentSize)
        ringtonesPopUpButton.setFrameOrigin(NSPoint(x: (view.frame.maxX / 2) - (ringtonesPopUpButton.intrinsicContentSize.width / 2), y: 100))
        ringtonesPopUpButton.action = #selector(self.ringtoneSelected)
        ringtonesPopUpButton.target = self
        
        makeEditable()
        
        view.addSubview(hourLabel)
        view.addSubview(minuteLabel)
        view.addSubview(secondLabel)
        view.addSubview(hourTextField)
        view.addSubview(minuteTextField)
        view.addSubview(secondTextField)
        view.addSubview(hourStepper)
        view.addSubview(minuteStepper)
        view.addSubview(secondStepper)
        view.addSubview(ringtonesPopUpButton)
    }
    
    func createTimer() {
        actualTimer = DispatchSource.makeTimerSource(flags: .strict, queue: queue)
        actualTimer.schedule(deadline: .now(), repeating: 1.0, leeway: .nanoseconds(0))
        actualTimer.setEventHandler {
            self.timeLeft = self.timer!.hoursLeft * 60 * 60 + self.timer!.minutesLeft * 60 + self.timer!.secondsLeft
            self.timeLeft -= 1
            
            if self.timeLeft < 0 {
                let notification = NSUserNotification()
                notification.identifier = "timer-"
                notification.title = "Clock"
                notification.subtitle = "Timer done"
                notification.hasActionButton = true
                notification.actionButtonTitle = "Repeat"
                notification.otherButtonTitle = "Dismiss"
                self.notificationCenter.deliver(notification)
                
                self.actualTimer.cancel()
            } else {
                self.timer!.hoursLeft = Int(floor(Double(self.timeLeft / 60 / 60)))
                self.timer!.minutesLeft = Int(floor(Double(self.timeLeft / 60))) - self.timer!.hoursLeft * 60
                self.timer!.secondsLeft = self.timeLeft - self.timer!.hoursLeft * 60 * 60 - self.timer!.minutesLeft * 60
                
                self.updateTextFields()
            }
        }
        
    }
    
    func updateTextFields() {
        DispatchQueue.main.async(execute: { () -> Void in
            self.hourTextField.stringValue = String(format: "%02d", self.timer!.hoursLeft)
            self.minuteTextField.stringValue = String(format: "%02d", self.timer!.minutesLeft)
            self.secondTextField.stringValue = String(format: "%02d", self.timer!.secondsLeft)
        })
    }
    
    override func mouseDown(with event: NSEvent) {
        // super ugly way to remove focus from text fields
        hourTextField.isEnabled = false
        hourTextField.isEnabled = true
        minuteTextField.isEnabled = false
        minuteTextField.isEnabled = true
        secondTextField.isEnabled = false
        secondTextField.isEnabled = true
    }
    
    @objc func ringtoneSelected(_ sender: NSPopUpButton) {
        ringtone = sender.titleOfSelectedItem!
        print(ringtone)
        defaults.set(ringtone, forKey: ringtoneKey)
    }
    
    @objc func hourStepperPressed(_ sender: NSStepper) {
        hours = Int(hourStepper.intValue)
        hourTextField.stringValue = String(format: "%02d", hours)
        defaults.set(hours, forKey: hoursKey)
    }
    @objc func minuteStepperPressed(_ sender: NSStepper) {
        minutes = Int(minuteStepper.intValue)
        minuteTextField.stringValue = String(format: "%02d", minutes)
        defaults.set(minutes, forKey: minutesKey)
    }
    @objc func secondStepperPressed(_ sender: NSStepper) {
        seconds = Int(secondStepper.intValue)
        secondTextField.stringValue = String(format: "%02d", seconds)
        defaults.set(seconds, forKey: secondsKey)
    }
    
    func makeEditable() {
        hourTextField.stringValue = String(format: "%02d", hours)
        minuteTextField.stringValue = String(format: "%02d", minutes)
        secondTextField.stringValue = String(format: "%02d", seconds)
        
        hourStepper.intValue = Int32(hours)
        minuteStepper.intValue = Int32(minutes)
        secondStepper.intValue = Int32(seconds)
        
        hourTextField.isEditable = true
        minuteTextField.isEditable = true
        secondTextField.isEditable = true
        
        hourStepper.isEnabled = true
        minuteStepper.isEnabled = true
        secondStepper.isEnabled = true
        
        ringtonesPopUpButton.isEnabled = true
    }
    func makeStatic() {
        hourTextField.isEditable = false
        minuteTextField.isEditable = false
        secondTextField.isEditable = false
        
        hourStepper.isEnabled = false
        minuteStepper.isEnabled = false
        secondStepper.isEnabled = false
        
        ringtonesPopUpButton.isEnabled = false
        
        if isTextField(inFocus: hourTextField) {
            hourTextField.window?.makeFirstResponder(nil)
        } else if isTextField(inFocus: minuteTextField) {
            minuteTextField.window?.makeFirstResponder(nil)
        } else if isTextField(inFocus: secondTextField) {
            secondTextField.window?.makeFirstResponder(nil)
        }
    }
    
    @IBAction func startPauseResumeBtnPressed(_ sender: Any) {
        switch timer!.currentState {
        case .off:
            makeStatic()
            timer = turnOnTimer(timer!)
        case .on:
            timer = pauseTimer(timer!)
        case .paused:
            timer = resumeTimer(timer!)
        }
        print(timer as Any)
    }
    @IBAction func cancelBtnPressed(_ sender: Any) {
        makeEditable()
        timer = turnOffTimer(timer!)
        print(timer as Any)
    }
    
    func turnOnTimer(_ timer: Timer) -> Timer {
        createTimer()
        actualTimer.resume()

        cancelBtn.isEnabled = true
        startPauseResumeBtn.title = "Pause"
        
        var newTimer = timer
        
        newTimer.hours = hours
        newTimer.minutes = minutes
        newTimer.seconds = seconds
        
        newTimer.hoursLeft = newTimer.hours
        newTimer.minutesLeft = newTimer.minutes
        newTimer.secondsLeft = newTimer.seconds

        newTimer.currentState = .on

        return newTimer
    }
    func turnOffTimer(_ timer: Timer) -> Timer {
        actualTimer.cancel()
        
        startPauseResumeBtn.title = "Start"
        cancelBtn.isEnabled = false
        
        var newTimer = timer
        
        newTimer.hours = hours
        newTimer.minutes = minutes
        newTimer.seconds = seconds

        newTimer.hoursLeft = newTimer.hours
        newTimer.minutesLeft = newTimer.minutes
        newTimer.secondsLeft = newTimer.seconds
        
        newTimer.currentState = .off
        
        return newTimer
    }
    func pauseTimer(_ timer: Timer) -> Timer {
        actualTimer.suspend()
        
        startPauseResumeBtn.title = "Resume"
        
        var newTimer = timer
        newTimer.currentState = .paused
        return newTimer
    }
    func resumeTimer(_ timer: Timer) -> Timer {
        actualTimer.resume()
        
        startPauseResumeBtn.title = "Pause"
        
        var newTimer = timer
        newTimer.currentState = .on
        return newTimer
    }
    
    // magic stackoverflow code to detect if a text field is in focus by MwcsMac https://stackoverflow.com/a/48530969
    func isTextField(inFocus textField: NSTextField) -> Bool {
        var inFocus = false
        inFocus = (textField.window?.firstResponder is NSTextView) && textField.window?.fieldEditor(false, for: nil) != nil && textField.isEqual(to: (textField.window?.firstResponder as? NSTextView)?.delegate)
        return inFocus
    }
    
    
    func getRingtones(path: String) -> [String] {
        let fileManager = FileManager.default
        var ringtones: [String] = []
        do {
            let tonePaths = try fileManager.contentsOfDirectory(atPath: path)
            for tp in tonePaths {
                let substringRingtone = tp.prefix(tp.count - 4)
                let stringRingtone = String(substringRingtone)
                ringtones.append(stringRingtone)
            }
        } catch let error as NSError {
            print("Error reading ringtones: \(error)")
        }
        return ringtones
    }
    
    // delegates
    
    func notificationDidDeliver(_ center: NSUserNotificationCenter, didDeliver notification: NSUserNotification) {
        if ringtone != "None" {
            let tone = URL(fileURLWithPath: "\(ringtonesPath)\(ringtone).m4r")
            do {
                audioPlayer = try AVAudioPlayer(contentsOf: tone)
                audioPlayer.prepareToPlay()
                audioPlayer.play()
                audioPlayer.numberOfLoops = -1
            } catch let e as NSError {
                print("something went wrong: \(e)")
            }
        }
    }
    func notificationDidActivate(_ center: NSUserNotificationCenter, didActivate notification: NSUserNotification) {
        if ringtone != "None" {
            audioPlayer.stop()
        }
        switch notification.activationType {
        case .actionButtonClicked:
            timer = turnOffTimer(timer!)
            timer = turnOnTimer(timer!)
            break;
        case .contentsClicked:
            //show popover
            break;
        default:
            break;
        }
    }
    func notificationDidDismissAlert(_ center: NSUserNotificationCenter, didDismissAlert notification: NSUserNotification) {
        notification.deliveryRepeatInterval?.second = nil
        notificationCenter.removeDeliveredNotification(notification)
        timer = turnOffTimer(timer!)
        makeEditable()
        if ringtone != "None" {
            audioPlayer.stop()
        }
    }
    
    func controlTextDidEndEditing(_ obj: Notification) {
        hourTextField.stringValue = String(format: "%02d", Int(hourTextField.stringValue) ?? 0)
        minuteTextField.stringValue = String(format: "%02d", Int(minuteTextField.stringValue) ?? 0)
        secondTextField.stringValue = String(format: "%02d", Int(secondTextField.stringValue) ?? 0)
        
        hours = Int(hourTextField.stringValue) ?? 0
        minutes = Int(minuteTextField.stringValue) ?? 0
        seconds = Int(secondTextField.stringValue) ?? 0
        
        defaults.set(hours, forKey: hoursKey)
        defaults.set(minutes, forKey: minutesKey)
        defaults.set(seconds, forKey: secondsKey)
    }
    
    func controlTextDidChange(_ obj: Notification) {
        let characterSet: NSCharacterSet = NSCharacterSet(charactersIn: "0123456789").inverted as NSCharacterSet
        self.hourTextField.stringValue = (self.hourTextField.stringValue.components(separatedBy: characterSet as CharacterSet) as NSArray).componentsJoined(by: "")
        self.minuteTextField.stringValue = (self.minuteTextField.stringValue.components(separatedBy: characterSet as CharacterSet) as NSArray).componentsJoined(by: "")
        self.secondTextField.stringValue = (self.secondTextField.stringValue.components(separatedBy: characterSet as CharacterSet) as NSArray).componentsJoined(by: "")
        
        if hourTextField.stringValue.count > 2 {
            hourTextField.stringValue = String(hourTextField.stringValue.prefix(2))
        }
        if minuteTextField.stringValue.count > 2 {
            minuteTextField.stringValue = String(minuteTextField.stringValue.prefix(2))
        }
        if secondTextField.stringValue.count > 2 {
            secondTextField.stringValue = String(secondTextField.stringValue.prefix(2))
        }
        
        if Int(hourTextField.stringValue) ?? 0 > 23 {
            hourTextField.stringValue = String(23)
        } else if Int(hourTextField.stringValue) ?? 0 < 0 {
            hourTextField.stringValue = String(0)
        }
        hourStepper.intValue = Int32(hourTextField.stringValue) ?? 0
        
        if Int(minuteTextField.stringValue) ?? 0 > 59 {
            minuteTextField.stringValue = String(59)
        } else if Int(minuteTextField.stringValue) ?? 0 < 0 {
            minuteTextField.stringValue = String(0)
        }
        minuteStepper.intValue = Int32(minuteTextField.stringValue) ?? 0
        
        if Int(secondTextField.stringValue) ?? 0 > 59 {
            secondTextField.stringValue = String(59)
        } else if Int(secondTextField.stringValue) ?? 0 < 0 {
            secondTextField.stringValue = String(0)
        }
        secondStepper.intValue = Int32(secondTextField.stringValue) ?? 0

    }
    
}

extension TimerViewController {
    static func freshController() -> TimerViewController {
        let storyboard = NSStoryboard(name: "Main", bundle: nil)
        let identifier = "TimerViewController"
        guard let timerViewController = storyboard.instantiateController(withIdentifier: identifier) as? TimerViewController else {
            fatalError("Why cant i find TimerViewController? - check Main.storyboard")
        }
        return timerViewController
    }
}
