//
//  StopwatchViewController.swift
//  Clock
//
//  Created by Dinosaur_Weirdo on 10/20/18.
//  Copyright © 2018 DinoW. All rights reserved.
//

import Cocoa

struct Lap {
    var name: String
    var time: Int
    
    init(name: String, time: Int) {
        self.name = name
        self.time = time
    }
}

class StopwatchViewController: NSViewController, NSTableViewDelegate, NSTableViewDataSource {
    enum state {
        case on, off, paused
    }
    
    var timeLbl = NSTextField(frame: NSRect(x: 0, y: 187, width: 450, height: 65))
    var swState: state = .off
    var laps: [Lap] = []
    var currentTime = 0
    
    var dst: DispatchSourceTimer!
    
    @IBOutlet var lapTableView: NSTableView!
    @IBOutlet var lapBtn: NSButton!
    @IBOutlet var startStopBtn: NSButton!
    
    override func viewDidLoad() {
        super.viewDidLoad()
        // Do view setup here.
        
        timeLbl.font = NSFont.monospacedDigitSystemFont(ofSize: 50.0, weight: .regular)
        timeLbl.isEditable = false
        timeLbl.isBezeled = false
        timeLbl.alignment = .center
        timeLbl.drawsBackground = false
        timeLbl.stringValue = "00:00.00"
        
        lapTableView.delegate = self
        lapTableView.dataSource = self
        
        lapBtn.title = "Lap"
        lapBtn.isEnabled = false
        startStopBtn.title = "Start"
        
        view.addSubview(timeLbl)
    }
    @IBAction func lapBtnPressed(_ sender: Any) {
        switch swState {
        case .on:
            createLap()
            break
        case .off:
            print("this should be impossible.") // this button is greyed out when swState is .off
            break
        case .paused:
            resetStopwatch()
            break
        }
    }
    @IBAction func startStopBtnPressed(_ sender: Any) {
        switch swState {
        case .on:
            pauseStopwatch()
            break
        case .off:
            startStopwatch()
            break
        case .paused:
            resumeStopwatch()
            break
        }
    }
    
    func createLap() {
        laps.insert(Lap(name: "Lap \(laps.count + 1)", time: 0), at: 0)
        lapTableView.reloadData()
        // just change laps[0] in the dispatch source timer
    }
    func resetStopwatch() {
        swState = .off
        lapBtn.title = "Lap"
        lapBtn.isEnabled = false
        startStopBtn.title = "Start"
        
        dst.cancel()
        dst.resume() // crashes if you cancel without resuming
        currentTime = 0
        laps.removeAll()
        updateTimeLbl()
        lapTableView.reloadData()
    }
    func pauseStopwatch() {
        swState = .paused
        lapBtn.title = "Reset"
        startStopBtn.title = "Start"
        
        dst.suspend()
    }
    func startStopwatch() {
        swState = .on
        lapBtn.title = "Lap"
        lapBtn.isEnabled = true
        startStopBtn.title = "Stop"
        
        laps.append(Lap(name: "Lap 1", time: 0))
        lapTableView.reloadData()
        
        createTimer()
        dst.resume()

    }
    func resumeStopwatch() {
        swState = .on
        lapBtn.title = "Lap"
        startStopBtn.title = "Stop"
        
        dst.activate()
    }
    
    func createTimer() {
        let queue = DispatchQueue(label: "com.dinow.Clock.stopwatch", qos: .userInteractive)
        dst = DispatchSource.makeTimerSource(flags: .strict, queue: queue)
        dst.schedule(deadline: .now(), repeating: 0.01, leeway: .nanoseconds(0))
        dst.setEventHandler {
            self.currentTime += 1
            self.laps[0].time += 1
            
            self.reloadEverything()
        }
    }
    
    func reloadEverything() {
        DispatchQueue.main.async(execute: { () -> Void in
            self.updateTimeLbl()
            
            self.lapTableView.reloadData(forRowIndexes: IndexSet(integer: 0), columnIndexes: IndexSet(integer: 1))
        })
    }
    
    func updateTimeLbl() {
        let min = Int(floor(Double(self.currentTime / 60 / 100)))
        let sec = Int(floor(Double(self.currentTime / 100))) - min * 60
        let mil = self.currentTime - min * 60 * 100 - sec * 100
        self.timeLbl.stringValue = "\(String(format: "%02d", min)):\(String(format: "%02d", sec)).\(String(format: "%02d", mil))"
    }
    
    // delegate
    
    func numberOfRows(in tableView: NSTableView) -> Int {
        return laps.count
    }
    func tableView(_ tableView: NSTableView, viewFor tableColumn: NSTableColumn?, row: Int) -> NSView? {
        let cellView = NSView()
        let label = NSTextField(frame: NSRect(x: 0, y: 0, width: 100, height: 17))
        label.isEditable = false
        label.isBezeled = false
        label.font = NSFont.monospacedDigitSystemFont(ofSize: 12, weight: .regular)
        label.drawsBackground = false
        cellView.addSubview(label)
        switch tableColumn {
        case lapTableView.tableColumns[0]:
            label.stringValue = laps[row].name
            break
        case lapTableView.tableColumns[1]:
            let unformattedTime = laps[row].time
            let min = Int(floor(Double(unformattedTime / 60 / 100)))
            let sec = Int(floor(Double(unformattedTime / 100))) - min * 60
            let mil = unformattedTime - min * 60 * 100 - sec * 100
            let formattedTime = "\(String(format:"%02d",min)):\(String(format:"%02d",sec)).\(String(format:"%02d",mil))"
            label.stringValue = formattedTime
            break
        default:
            print("weird table column: \(tableColumn!)")
            break
        }
        return cellView
    }
    func tableView(_ tableView: NSTableView, heightOfRow row: Int) -> CGFloat {
        return 20
    }
}

extension StopwatchViewController {
    static func freshController() -> StopwatchViewController {
        let storyboard = NSStoryboard(name: "Main", bundle: nil)
        let identifier = "StopwatchViewController"
        guard let stopwatchViewController = storyboard.instantiateController(withIdentifier: identifier) as? StopwatchViewController else {
            fatalError("Why cant i find StopwatchViewController? - check Main.storyboard")
        }
        return stopwatchViewController
    }
}

/* TODO/BUGS
 * - stopwatch eats up cpu when running, like 40-50% cpu usage is spent on it
 */
