//
//  AlarmViewController.swift
//  Clock
//
//  Created by Dinosaur_Weirdo on 6/25/18.
//  Copyright © 2018 DinoW. All rights reserved.
//

import Cocoa
import AVFoundation

struct Alarm: Codable {
    var hour: Int
    var minute: Int
    var ringtone: String
    var snooze: Int
    var isSnoozed: Bool
    var label: String
    var repeatDays: [Bool]
    var enabled: Bool
    
    init(ringtones: [String]) {
        //set default values
        let date = Date()
        let calendar = Calendar.current
        let cHour = calendar.component(.hour, from: date)
        self.hour = (cHour + 1) % 24
        self.minute = 0
        if ringtones.contains("Radar") {
            self.ringtone = "Radar"
        } else {
            self.ringtone = ringtones[0]
        }
        self.snooze = 9
        self.isSnoozed = false
        self.label = "Alarm"
        self.repeatDays = [false,false,false,false,false,false,false]
        self.enabled = true
    }
    
    init(hour: Int, minute: Int, ringtone: String, snooze: Int, label: String, repeatDays: [Bool], enabled: Bool) {
        self.hour = hour
        self.minute = minute
        self.ringtone = ringtone
        self.snooze = snooze
        self.isSnoozed = false
        self.label = label
        self.repeatDays = repeatDays
        self.enabled = enabled
    }
    
    init(fromDictionary dict: Dictionary<String, Any>) {
        self.hour = dict["hour"] as! Int
        self.minute = dict["minute"] as! Int
        self.ringtone = dict["ringtone"] as! String
        self.snooze = dict["snooze"] as! Int
        self.isSnoozed = false
        self.label = dict["label"] as! String
        self.repeatDays = dict["repeatDays"] as! [Bool]
        self.enabled = dict["enabled"] as! Bool
    }
}


class AlarmViewController: NSViewController, NSTableViewDataSource, NSTableViewDelegate, NSTextFieldDelegate, NSUserNotificationCenterDelegate {
    
    var do24HourTime = UserDefaults.standard.bool(forKey: "com.DinoW.Do24HourTime")
    
    var alarms: [Alarm] = []
    var alarmViews: [NSView] = []
    var alarmNotifications: [[NSUserNotification?]] = []
    let notificationCenter = AppDelegate.notificationCenter
    var audioPlayer = AVAudioPlayer()
    var currentlyBeingEdited = -1
    let ringtonesPath = "/System/Library/PrivateFrameworks/ToneLibrary.framework/Versions/A/Resources/Ringtones/"
    
    // theres only one editable alarm view at a time so we can define all this shit up here rather than in makeEditable
    let alarmTimeDatePicker = NSDatePicker(frame: NSRect(x: 8, y: 34, width: 85, height: 27))
    
    let fakeActivationBtn = NSButton(checkboxWithTitle: "", target: self, action: nil)
    
    let alarmLabelTextField = NSTextField(frame: NSRect(x: 97, y: 34, width: 76, height: 23))
    
    let ringtonePopUpBtn = NSPopUpButton(frame: NSRect(x: 175, y: 34, width: 76, height: 26))
    
    let snoozeTextField = NSTextField(frame: NSRect(x: 301, y: 34, width: 24, height: 23))
    let snoozeStepper = NSStepper(frame: NSRect(x: 325, y: 33, width: 19, height: 27))
    let snoozeLbl = NSTextField(frame: NSRect(x: 343, y: 35, width: 31, height: 17))
    
    let doneBtn = NSButton(frame: NSRect(x: 376, y: 29, width: 46, height: 32))
    let deleteBtn = NSButton(frame: NSRect(x: 3, y: -3, width: 68, height: 32))
    let cancelBtn = NSButton(frame: NSRect(x: 410, y: -10, width: 42, height: 42))
    
    let repeatSun = NSButton(checkboxWithTitle: "Sun", target: self, action: nil)
    let repeatMon = NSButton(checkboxWithTitle: "Mon", target: self, action: nil)
    let repeatTue = NSButton(checkboxWithTitle: "Tue", target: self, action: nil)
    let repeatWed = NSButton(checkboxWithTitle: "Wed", target: self, action: nil)
    let repeatThu = NSButton(checkboxWithTitle: "Thu", target: self, action: nil)
    let repeatFri = NSButton(checkboxWithTitle: "Fri", target: self, action: nil)
    let repeatSat = NSButton(checkboxWithTitle: "Sat", target: self, action: nil)
    
    func newAlarmView(row: Int) {
        if (currentlyBeingEdited > -1) {
            alarmViews[currentlyBeingEdited] = makeStatic(row: currentlyBeingEdited)
        }
        alarmViews.insert(makeEditable(row: row), at: row)
        
        tableView.reloadData()
    }
    
    func makeEditable(row: Int) -> NSView {
        currentlyBeingEdited = row
        
        let alarm = alarms[row]
        let editableAlarmView = NSView(frame: NSRect(origin: .zero, size: CGSize(width: 450, height: 60)))
        
        doneBtn.title = "Done"
        doneBtn.bezelStyle = .roundRect
        
        deleteBtn.title = "Delete"
        deleteBtn.bezelStyle = .roundRect
        
        cancelBtn.title = "X"
        cancelBtn.bezelStyle = .circular
        cancelBtn.alignment = .center
        
        fakeActivationBtn.frame = NSRect(x: 424, y: 36, width: 18, height: 18)
        fakeActivationBtn.state = .on
        fakeActivationBtn.isEnabled = false
        
        alarmTimeDatePicker.datePickerMode = .single
        alarmTimeDatePicker.datePickerStyle = .textFieldAndStepper
        alarmTimeDatePicker.datePickerElements = .hourMinute
        alarmTimeDatePicker.dateValue = Date(timeIntervalSinceReferenceDate: Double(alarm.hour * 3600 + alarm.minute * 60 - TimeZone.current.secondsFromGMT()))
        
        alarmLabelTextField.placeholderString = "Alarm label"
        alarmLabelTextField.stringValue = alarm.label
        alarmLabelTextField.usesSingleLineMode = true
        
        repeatSun.frame = NSRect(x: 72,  y: 4, width: 50, height: 18)
        repeatMon.frame = NSRect(x: 122, y: 4, width: 50, height: 18)
        repeatTue.frame = NSRect(x: 175, y: 4, width: 50, height: 18)
        repeatWed.frame = NSRect(x: 224, y: 4, width: 50, height: 18)
        repeatThu.frame = NSRect(x: 278, y: 4, width: 50, height: 18)
        repeatFri.frame = NSRect(x: 328, y: 4, width: 50, height: 18)
        repeatSat.frame = NSRect(x: 370, y: 4, width: 50, height: 18)
 
        repeatSun.state = alarm.repeatDays[0] ? .on : .off
        repeatMon.state = alarm.repeatDays[1] ? .on : .off
        repeatTue.state = alarm.repeatDays[2] ? .on : .off
        repeatWed.state = alarm.repeatDays[3] ? .on : .off
        repeatThu.state = alarm.repeatDays[4] ? .on : .off
        repeatFri.state = alarm.repeatDays[5] ? .on : .off
        repeatSat.state = alarm.repeatDays[6] ? .on : .off
        
        ringtonePopUpBtn.addItems(withTitles: getRingtones(path: ringtonesPath).sorted())
        ringtonePopUpBtn.insertItem(withTitle: "None", at: 0)
        ringtonePopUpBtn.setFrameSize(ringtonePopUpBtn.intrinsicContentSize)
        ringtonePopUpBtn.selectItem(withTitle: alarm.ringtone)
        
        snoozeTextField.stringValue = String(alarm.snooze)
        snoozeTextField.isEditable = true
        snoozeTextField.delegate = self
        
        snoozeStepper.intValue = Int32(alarm.snooze)
        snoozeStepper.autorepeat = true
        snoozeStepper.valueWraps = false
        snoozeStepper.maxValue = 99
        snoozeStepper.minValue = 1
        snoozeStepper.increment = 1
        snoozeStepper.target = self
        snoozeStepper.action = #selector(AlarmViewController.snoozeStepperPressed)
        
        snoozeLbl.stringValue = "mins"
        snoozeLbl.isEditable = false
        snoozeLbl.isBezeled = false
        snoozeLbl.drawsBackground = false
        
        editableAlarmView.addSubview(doneBtn)
        editableAlarmView.addSubview(deleteBtn)
        editableAlarmView.addSubview(cancelBtn)
        editableAlarmView.addSubview(fakeActivationBtn)
        editableAlarmView.addSubview(alarmTimeDatePicker)
        editableAlarmView.addSubview(alarmLabelTextField)
        editableAlarmView.addSubview(repeatSun)
        editableAlarmView.addSubview(repeatMon)
        editableAlarmView.addSubview(repeatTue)
        editableAlarmView.addSubview(repeatWed)
        editableAlarmView.addSubview(repeatThu)
        editableAlarmView.addSubview(repeatFri)
        editableAlarmView.addSubview(repeatSat)
        editableAlarmView.addSubview(ringtonePopUpBtn)
        editableAlarmView.addSubview(snoozeTextField)
        editableAlarmView.addSubview(snoozeStepper)
        editableAlarmView.addSubview(snoozeLbl)
        
        return editableAlarmView
    }
    func makeStatic(row: Int) -> NSView {
        currentlyBeingEdited = -1
        writeJSONFromAlarms()
        
        let alarm = alarms[row]
        let staticAlarmView = NSView(frame: NSRect(x: 0, y: 0, width: 450, height: 30))
        
        let editBtn = NSButton(frame: NSRect(x: 376, y: -1, width: 46, height: 32))
        editBtn.title = "Edit"
        editBtn.bezelStyle = .roundRect
        editBtn.tag = row
        editBtn.target = self
        editBtn.action = #selector(AlarmViewController.editBtnPressed)
        
        let activationBtn = NSButton(checkboxWithTitle: "", target: self, action: #selector(toggleAlarm(_:)))
        activationBtn.frame = NSRect(x: 424, y: 6, width: 18, height: 18)
        activationBtn.state = alarm.enabled ? .on : .off
        activationBtn.tag = row
        
        let timeLbl = NSTextField(frame: NSRect(x: 10, y: 7, width: 55, height: 17))
        timeLbl.stringValue = do24HourTime ? ("\(String(format: "%02d", alarm.hour)):\(String(format: "%02d", alarm.minute))") : ((alarm.hour % 12 == 0 ? "12" : String(alarm.hour % 12)) + ":" + String(format: "%02d", alarm.minute) + (alarm.hour < 12 ? "am" : "pm"))
        timeLbl.isEditable = false
        timeLbl.isBezeled = false
        timeLbl.sizeToFit()
        timeLbl.drawsBackground = false
        
        let alarmLabelLbl = NSTextField(frame: NSRect(x: 65, y: 5, width: 90, height: 17))
        alarmLabelLbl.stringValue = alarm.label
        alarmLabelLbl.isEditable = false
        alarmLabelLbl.isBezeled = false
        alarmLabelLbl.usesSingleLineMode = true
//        alarmLabelLbl.drawsBackground = false
        
        let ringtoneLbl = NSTextField(frame: NSRect(x: 165, y: 7, width: 40, height: 17))
        ringtoneLbl.stringValue = alarm.ringtone
        ringtoneLbl.isEditable = false
        ringtoneLbl.isBezeled = false
        ringtoneLbl.sizeToFit()
        ringtoneLbl.drawsBackground = false
        
        let snoozeLbl = NSTextField(frame: NSRect(x: 265, y: 7, width: 40, height: 17))
        if alarm.isSnoozed {
            let c = Calendar.current
            let d = Date()
            let weekday = c.component(.weekday, from: d)
            let notification = alarmNotifications[row][alarmNotifications[row].count > 1 ? weekday - 1 : 0]
            let deliveryDate = notification?.deliveryDate
            let interval = c.dateComponents([.minute], from: d, to: deliveryDate!)
//            print("interval: \(interval)")
//            print("deliveryDate: \(deliveryDate)")
//            print("date: \(d)")
            snoozeLbl.stringValue = "\(interval.minute! + 1)m left"
        } else {
            snoozeLbl.stringValue = "Snooze: " + String(alarm.snooze) + " mins"
        }
        snoozeLbl.font = NSFont.monospacedDigitSystemFont(ofSize: 12, weight: .regular)
        snoozeLbl.isEditable = false
        snoozeLbl.isBezeled = false
        snoozeLbl.sizeToFit()
        snoozeLbl.drawsBackground = false
        
        if alarm.isSnoozed {
            let snoozeStopBtn = NSButton(frame: NSRect(x: 325, y: -1, width: 40, height: 32))
            snoozeStopBtn.title = "Stop"
            snoozeStopBtn.bezelStyle = .roundRect
            snoozeStopBtn.tag = row
            snoozeStopBtn.target = self
            snoozeStopBtn.action = #selector(stopSnooze(_:))
            staticAlarmView.addSubview(snoozeStopBtn)
        }
        
        staticAlarmView.addSubview(editBtn)
        staticAlarmView.addSubview(activationBtn)
        staticAlarmView.addSubview(timeLbl)
        staticAlarmView.addSubview(alarmLabelLbl)
        staticAlarmView.addSubview(ringtoneLbl)
        staticAlarmView.addSubview(snoozeLbl)
        
        return staticAlarmView
    }
    
    func createNotification(row: Int, isPlaceholder: Bool) -> [NSUserNotification?] {
        let alarm = alarms[row]
        
        var notificationsArray: [NSUserNotification?] = []
        if alarm.repeatDays.contains(true) {
            for (day,doesRepeat) in alarm.repeatDays.enumerated() {
                if doesRepeat {
                    if !isPlaceholder {
                        let alarm = alarms[row]
                        
                        let notification = NSUserNotification()
                        notification.identifier = "alarm-\(row)-\(day)"
                        notification.title = "Clock"
                        notification.subtitle = alarm.label
                        let calendar = Calendar.current
                        var dc = DateComponents()
                        let date = Date()
                        dc.timeZone = TimeZone.autoupdatingCurrent
                        dc.hour = alarm.hour
                        dc.minute = alarm.minute
                        if calendar.component(.weekday, from: date) == day + 1 {
//                            print("\(day): a")
                            let timeInMinutes = calendar.dateComponents([.hour], from: date).hour! * 60 + calendar.dateComponents([.minute], from: date).minute!
                            dc.day = alarm.hour * 60 + alarm.minute > timeInMinutes ? calendar.component(.day, from: date) : calendar.component(.day, from: calendar.date(byAdding: .day, value: 1, to: date)!)
                        } else {
//                            print("\(day): b")
                            let weekdayComponents = DateComponents(calendar: calendar, weekday: day + 1)
                            let nextWeekday = calendar.nextDate(after: date, matching: weekdayComponents, matchingPolicy: .nextTimePreservingSmallerComponents)
                            dc.day = calendar.component(.day, from: nextWeekday!)
                        }
                        dc.month = calendar.component(.month, from: date)
                        dc.year = calendar.component(.year, from: date)
                        notification.deliveryDate = alarm.enabled ? calendar.date(from: dc) : nil
                        notification.hasActionButton = true
                        notification.actionButtonTitle = "Snooze"
                        notification.otherButtonTitle = "Dismiss"
                        notification.deliveryRepeatInterval?.day = 7
                        if alarm.enabled {
                            notificationCenter.scheduleNotification(notification)
                        }
                        notificationsArray.append(notification)
                    } else {
                        notificationsArray.append(nil)
                    }
                    
                } else {
                    notificationsArray.append(nil)
                }
            }
        } else {
            if !isPlaceholder {
                let alarm = alarms[row]
                
                let notification = NSUserNotification()
                notification.identifier = "alarm-\(row)-0" //all the days that it repeats have the same identifier so that notification center deletes them if they double up
                notification.title = "Clock"
                notification.subtitle = alarm.label
                let calendar = Calendar.current
                var dc = DateComponents()
                let date = Date()
                let timeInMinutes = calendar.dateComponents([.hour], from: date).hour! * 60 + calendar.dateComponents([.minute], from: date).minute!
                dc.timeZone = TimeZone.autoupdatingCurrent
                dc.hour = alarm.hour
                dc.minute = alarm.minute
                dc.day = alarm.hour * 60 + alarm.minute > timeInMinutes ? calendar.component(.day, from: date) : calendar.component(.day, from: calendar.date(byAdding: .day, value: 1, to: date)!)
                dc.month = calendar.component(.month, from: date)
                dc.year = calendar.component(.year, from: date)
                notification.deliveryDate = alarm.enabled ? calendar.date(from: dc) : nil
                notification.hasActionButton = true
                notification.actionButtonTitle = "Snooze"
                notification.otherButtonTitle = "Dismiss"
                if alarm.enabled {
                    notificationCenter.scheduleNotification(notification)
                }
                    
                notificationsArray.append(notification)
            } else {
                notificationsArray.append(nil)
            }

        }
        return notificationsArray
    }
    
    func controlTextDidChange(_ obj: Notification) {
        // yes, this runs whenever any textfield is edited; yes, that might cause issues later. ask me if i give a fuck.
        let characterSet: NSCharacterSet = NSCharacterSet(charactersIn: "0123456789").inverted as NSCharacterSet
        self.snoozeTextField.stringValue = (self.snoozeTextField.stringValue.components(separatedBy: characterSet as CharacterSet) as NSArray).componentsJoined(by: "")
        if snoozeTextField.stringValue.count > 2 {
            snoozeTextField.stringValue = String(snoozeTextField.stringValue.prefix(2))
        }
        snoozeStepper.intValue = Int32(snoozeTextField.stringValue) ?? 0
    }
    
    func reorderAlarms() {
        var alarmsNew: [Alarm] = []
        for a in alarms {
            if alarmsNew.count > 0 {
                let timeInMinutes = a.hour * 60 + a.minute
                var didGetInserted = false // this is slightly ugly, i don't care. fight me.
                for (i,current) in alarmsNew.enumerated() {
                    if current.hour * 60 + current.minute > timeInMinutes {
                        alarmsNew.insert(a, at: i)
                        didGetInserted = true
                        break
                    }
                }
                if !didGetInserted {
                    alarmsNew.append(a)
                }
            } else {
                alarmsNew.append(a)
            }
        }
        
        alarms = alarmsNew
//        for a in alarms {
//            print(a)
//        }
        
        for i in 0..<alarms.count {
            alarmViews[i] = makeStatic(row: i)
            alarmNotifications[i] = createNotification(row: i, isPlaceholder: false)
        }
    }
    
    @objc func snoozeStepperPressed(_ sender: NSStepper) {
        snoozeTextField.stringValue = String(snoozeStepper.intValue)
    }
    @objc func stopSnooze(_ sender: NSButton) {
        let row = sender.tag
        if let n = getNotificationFrom(identifier: "\(row)-") {
            notificationCenter.removeScheduledNotification(n)
        }
        if !alarms[row].repeatDays.contains(true) {
            alarms[row].enabled = false
        }
        alarms[row].isSnoozed = false
        alarmViews[row] = makeStatic(row: row)
        writeJSONFromAlarms()
        updateAlarmNotificationsFromAlarms()
        tableView.reloadData()
    }
    @objc func toggleAlarm(_ sender: NSButton) {
        let row = sender.tag
        alarms[row].enabled = sender.state == .on
        if alarms[row].enabled {
            alarmNotifications[row] = createNotification(row: row, isPlaceholder: false)
        } else {
            if let n = getNotificationFrom(identifier: "\(row)-") { // fancy if let statement isn't fooling anyone, this code is still fucking ugly lol
                notificationCenter.removeScheduledNotification(n)
            }
        }
    }
    @objc func editBtnPressed(_ sender: NSButton) {
        let row = sender.tag
        alarmViews[row] = makeEditable(row: row)
        
        tableView.reloadData()
    }
    @objc func doneBtnPressed() {
        let newAlarmDateTime = alarmTimeDatePicker.dateValue
        let newAlarmLabel = alarmLabelTextField.stringValue == "" ? "Alarm" : alarmLabelTextField.stringValue
        let newAlarmRingtone = ringtonePopUpBtn.titleOfSelectedItem!
        let newAlarmSnooze = snoozeTextField.stringValue == "" ? 9 : Int(snoozeTextField.stringValue)!
        let newAlarmRepeatDays = [repeatSun.state == .on,
                                  repeatMon.state == .on,
                                  repeatTue.state == .on,
                                  repeatWed.state == .on,
                                  repeatThu.state == .on,
                                  repeatFri.state == .on,
                                  repeatSat.state == .on]
        let newAlarmEnabled = true
        let calendar = Calendar.current
        
        let alarmToReplaceOldAlarm = Alarm(hour: calendar.component(.hour, from: newAlarmDateTime),
                                           minute: calendar.component(.minute, from: newAlarmDateTime),
                                           ringtone: newAlarmRingtone,
                                           snooze: newAlarmSnooze,
                                           label: newAlarmLabel,
                                           repeatDays: newAlarmRepeatDays,
                                           enabled: newAlarmEnabled)

        for i in 0..<7 {
            let n = getNotificationFrom(identifier: "\(currentlyBeingEdited)-\(i)")
            if n != nil {
                notificationCenter.removeDeliveredNotification(n!)
            }
        }
        
        alarms[currentlyBeingEdited] = alarmToReplaceOldAlarm
        alarmNotifications[currentlyBeingEdited] = createNotification(row: currentlyBeingEdited, isPlaceholder: false)
        alarmViews[currentlyBeingEdited] = makeStatic(row: currentlyBeingEdited)
        
        reorderAlarms()
        
//        print("--")
//        print(notificationCenter.deliveredNotifications)
//        print(notificationCenter.scheduledNotifications)
//        print(alarmNotifications)
        
        tableView.reloadData()
    }
    @objc func deleteBtnPressed() {
        alarms.remove(at: currentlyBeingEdited)
        alarmViews.remove(at: currentlyBeingEdited)
        for i in 0..<7 {
            let n = getNotificationFrom(identifier: "\(currentlyBeingEdited)-\(i)")
            if n != nil {
                notificationCenter.removeDeliveredNotification(n!)
            }
        }
        alarmNotifications.remove(at: currentlyBeingEdited)
        
        currentlyBeingEdited = -1
        
        reorderAlarms()
        writeJSONFromAlarms()
        
        tableView.reloadData()
    }
    @objc func cancelBtnPressed() {
        alarmViews[currentlyBeingEdited] = makeStatic(row: currentlyBeingEdited)
        
        tableView.reloadData()
    }
    

    func getNotificationFrom(identifier: String) -> NSUserNotification? {
//        alarmNotifications is structured as a two dimensional array of NSUserNotification? with 7 columns if theres a repeat and 1 column if there isn't
//        [[NSUserNotification?,NSUserNotification?,NSUserNotification?,NSUserNotification?,NSUserNotification?,NSUserNotification?,NSUserNotification?],
//         [NSUserNotification?,NSUserNotification?,NSUserNotification?,NSUserNotification?,NSUserNotification?,NSUserNotification?,NSUserNotification?],
//         [NSUserNotification?],
//         [NSUserNotification?,NSUserNotification?,NSUserNotification?,NSUserNotification?,NSUserNotification?,NSUserNotification?,NSUserNotification?]]
//        where some in the 7-column ones will be nil (so it doesn't repeat on that day) but NONE of the 1-column ones will be nil
//        however, they may be placeholders, and we need a way to check for that
//        we could just make another array called alarmPlaceholders and fill it with Bools indicating whether or not that notification is a placeholder
//        but i want something more elegant than that
//        then again, 50% of this code is the exact opposite of elegant, and it's not like i'm known for coming up with elegant code, so fuck it and fuck you
        print(identifier.prefix(upTo: identifier.firstIndex(of: Character("-"))!))
        let intIdentifier = Int(identifier.prefix(upTo: identifier.firstIndex(of: Character("-"))!))!
        let current = Calendar.current
        let date = Date()
        let weekday = current.component(.weekday, from: date)
        return alarmNotifications[intIdentifier][alarmNotifications[intIdentifier].count > 1 ? weekday - 1 : 0] //nevermind i can just make it optional and throw an if around it
    }
    
    @IBAction func clearAlarms(_ sender: Any) {
        alarmViews.removeAll()
        alarms.removeAll()
        alarmNotifications.removeAll()
        notificationCenter.removeAllDeliveredNotifications()
        currentlyBeingEdited = -1
        
        writeJSONFromAlarms()
        
        tableView.reloadData()
    }
    @IBAction func newAlarmButtonClicked(_ sender: Any) {
        let ringtones = getRingtones(path: ringtonesPath)
        
        let newAlarm = Alarm(ringtones: ringtones)
//        print(newAlarm.hour)
//        print(newAlarm.minute)
//        print(newAlarm.ringtone)
//        print(newAlarm.snooze)
//        print(newAlarm.label)
//        print(newAlarm.repeatDays)
//        print(newAlarm.enabled)
        
        let newAlarmLocation = insertAlarm(alarm: newAlarm)
        
        newAlarmView(row: newAlarmLocation)
        alarmNotifications.insert(createNotification(row: newAlarmLocation, isPlaceholder: true), at: newAlarmLocation)
    }
    
    @IBOutlet var tableView: NSTableView!
    
    override func viewDidAppear() {
        do24HourTime = UserDefaults.standard.bool(forKey: "com.DinoW.Do24HourTime")
        
        updateAlarmViewsFromAlarms()
        tableView.reloadData()
    }
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        tableView.delegate = self
        tableView.dataSource = self
        tableView.selectionHighlightStyle = .none
        
//        notificationCenter.delegate = self
//        notificationCenter.removeAllDeliveredNotifications()
//        notificationCenter.scheduledNotifications = []
        
        doneBtn.target = self
        doneBtn.action = #selector(AlarmViewController.doneBtnPressed)
        deleteBtn.target = self
        deleteBtn.action = #selector(AlarmViewController.deleteBtnPressed)
        cancelBtn.target = self
        cancelBtn.action = #selector(AlarmViewController.cancelBtnPressed)
        
        getAlarmsFromJSON()
        updateAlarmViewsFromAlarms()
        updateAlarmNotificationsFromAlarms()
        
        //delete header
        tableView.headerView = nil
    }
    
    func getAlarmsFromJSON() {
        let alarmFilePath = "\(NSHomeDirectory())/Library/Application Support/Clock/alarms.json"
        if !FileManager.default.fileExists(atPath: alarmFilePath) {
            createAlarmsFile()
        }
        do {
            let file = URL(fileURLWithPath: alarmFilePath)
            let data = try Data(contentsOf: file)
            let json = try JSONSerialization.jsonObject(with: data, options: [])
            if let object = json as? [Any?] {
                // json should be an array
                for jsonAlarm in object {
                    let alarm = Alarm(fromDictionary: jsonAlarm as! Dictionary<String, Any>)
                    alarms.append(alarm)
                }
            } else {
                print("json is invalid")
            }
        } catch {
            print(error.localizedDescription)
        }
    }
    
    func createAlarmsFile() {
        let fileManager = FileManager.default
        let dirPath =  "\(NSHomeDirectory())/Library/Application Support/Clock/"
        if !fileManager.fileExists(atPath: dirPath) {
            do {
                try fileManager.createDirectory(atPath: dirPath, withIntermediateDirectories: true, attributes: nil)
            } catch {
                print("Couldn't create directory")
            }
        }
        let filePath = "\(NSHomeDirectory())/Library/Application Support/Clock/alarms.json"
        if !fileManager.fileExists(atPath: filePath) {
            try? "[]".write(to: URL(fileURLWithPath: filePath), atomically: true, encoding: .utf8)
        }
    }
    
    func writeJSONFromAlarms() {
        if let encodedData = try? JSONEncoder().encode(alarms) {
            let path = "\(NSHomeDirectory())/Library/Application Support/Clock/alarms.json"
            let pathAsURL = URL(fileURLWithPath: path)
            do {
                try encodedData.write(to: pathAsURL)
            }
            catch let error as NSError {
                print("Failed to write JSON data: \(error.localizedDescription)")
            }
        }
    }
    
    func updateAlarmViewsFromAlarms() {
        alarmViews.removeAll()
        for i in 0..<alarms.count {
            alarmViews.append(makeStatic(row: i))
        }
        
        tableView.reloadData()
    }
    
    func updateAlarmNotificationsFromAlarms() {
        alarmNotifications.removeAll()
        for i in 0..<alarms.count {
            alarmNotifications.append(createNotification(row: i, isPlaceholder: false))
        }
    }
    
    func insertAlarm(alarm: Alarm) -> Int {
        let alarmTimeInMinutes = alarm.hour * 60 + alarm.minute
        for (i,a) in alarms.enumerated() {
            let timeInMinutes = a.hour * 60 + a.minute
            if (timeInMinutes > alarmTimeInMinutes) {
                alarms.insert(alarm, at: i)
                return i
            }
        }
        alarms.append(alarm)
        return alarms.count - 1
    }
    
    func getRingtones(path: String) -> [String] {
        let fileManager = FileManager.default
        var ringtones: [String] = []
        do {
            let tonePaths = try fileManager.contentsOfDirectory(atPath: path)
            for tp in tonePaths {
                let substringRingtone = tp.prefix(tp.count - 4)
                let stringRingtone = String(substringRingtone)
                ringtones.append(stringRingtone)
            }
        } catch let error as NSError {
            print("Error reading ringtones: \(error)")
        }
        return ringtones
    }
    
    
    // delegate methods
    
    public func notificationDidDeliver(_ center: NSUserNotificationCenter, didDeliver notification: NSUserNotification) {
        let notificationType = notification.identifier!.prefix(upTo: notification.identifier!.firstIndex(of: Character("-"))!)
        if notificationType == "alarm" {
            let index = notification.identifier!.index(notification.identifier!.startIndex, offsetBy: 6)
            let alarmID = String(notification.identifier![index...])
            let row = Int(alarmID.prefix(upTo: alarmID.firstIndex(of: Character("-"))!))!
            let alarm = alarms[row]
            if alarm.ringtone != "None" {
                let ringtone = URL(fileURLWithPath: "\(ringtonesPath)\(alarm.ringtone).m4r")
                //            print(ringtone)
                //            let sound = NSSound(contentsOfFile: ringtone, byReference: true)
                do {
                    audioPlayer = try AVAudioPlayer(contentsOf: ringtone)
                    audioPlayer.prepareToPlay()
                    audioPlayer.play()
                    audioPlayer.numberOfLoops = -1
                } catch let e as NSError {
                    print("something went wrong: \(e)")
                }
            }
        }
    }
    public func notificationDidActivate(_ center: NSUserNotificationCenter, didActivate notification: NSUserNotification) {
//        print("didActivate")
        let notificationType = notification.identifier!.prefix(upTo: notification.identifier!.firstIndex(of: Character("-"))!)
        if notificationType == "alarm" {
            let index = notification.identifier!.index(notification.identifier!.startIndex, offsetBy: 6)
            let alarmID = String(notification.identifier![index...])
            let row = Int(alarmID.prefix(upTo: alarmID.firstIndex(of: Character("-"))!))!
            audioPlayer.stop()
            switch notification.activationType {
            case .actionButtonClicked:
//                print("snoozed")
//                notification.deliveryRepeatInterval?.minute = alarms[row].snooze //TODO: test without this
                notification.deliveryDate = Date(timeIntervalSinceNow: Double(alarms[row].snooze * 60))
                let weekday = Calendar.current.component(.weekday, from: Date())
                alarmNotifications[row][alarmNotifications[row].count > 1 ? weekday - 1 : 0] = notification
                notificationCenter.scheduleNotification(notification)
                alarms[row].isSnoozed = true
                alarmViews[row] = makeStatic(row: row)
                print(alarms[row].isSnoozed)
                print(Date())
                break;
            case .contentsClicked:
//                print("contents clicked")
//show popover
                break;
            default:
//            print("other")
                break;
            }
        }
    }
    public func notificationDidDismissAlert(_ center: NSUserNotificationCenter, didDismissAlert notification: NSUserNotification) {
        //        print("dismissed")
        let notificationType = notification.identifier!.prefix(upTo: notification.identifier!.firstIndex(of: Character("-"))!)
        if notificationType == "alarm" {
            let index = notification.identifier!.index(notification.identifier!.startIndex, offsetBy: 6)
            let alarmID = String(notification.identifier![index...])
            let row = Int(alarmID.prefix(upTo: alarmID.firstIndex(of: Character("-"))!))!
//            notification.deliveryRepeatInterval?.minute = nil
            notificationCenter.removeDeliveredNotification(notification)
            if currentlyBeingEdited != row {
                if !alarms[row].repeatDays.contains(true) {
                    alarms[row].enabled = false
                }
                audioPlayer.stop()
                alarms[row].isSnoozed = false
                alarmViews[row] = makeStatic(row: row)
                writeJSONFromAlarms()
                tableView.reloadData()
            }
        }

    }
    

    // NSTableView
    func numberOfRows(in tableView: NSTableView) -> Int {
        return alarmViews.count
    }
    func tableView(_ tableView: NSTableView, viewFor tableColumn: NSTableColumn?, row: Int) -> NSView? {
        return alarmViews[row]
    }
    func tableView(_ tableView: NSTableView, heightOfRow row: Int) -> CGFloat {
        // editable alarm views are bigger than static alarm views
        if row == currentlyBeingEdited {
            return CGFloat(60)
        } else {
            return CGFloat(30)
        }
    }

    
}
    
extension AlarmViewController {
    static func freshController() -> AlarmViewController {
        let storyboard = NSStoryboard(name: "Main", bundle: nil)
        let identifier = "AlarmViewController"
        guard let alarmViewController = storyboard.instantiateController(withIdentifier: identifier) as? AlarmViewController else {
            fatalError("Why cant i find AlarmViewController? - check Main.storyboard")
        }
        return alarmViewController
    }
}
